function [sensor_data, field_data] = kspaceFirstOrder1D(kgrid, medium, source, sensor, varargin)
%KSPACEFIRSTORDER1D     1D time-domain simulation of wave propagation.
%
% DESCRIPTION:
%       kspaceFirstOrder1D simulates the time-domain propagation of linear
%       compressional waves through a one-dimensional homogeneous or
%       heterogeneous acoustic medium given four input structures: kgrid,
%       medium, source, and sensor. The computation is based on a
%       first-order k-space model which allows power law absorption and a
%       heterogeneous sound speed and density. At each time-step (defined
%       by kgrid.t_array), the pressure at the positions defined by
%       sensor.mask are recorded and stored. If kgrid.t_array is set to
%       'auto', this array is automatically generated using makeTime. An
%       absorbing boundary condition called a perfectly matched layer (PML)
%       is implemented to prevent waves that leave one side of the domain
%       being reintroduced from the opposite side (a consequence of using
%       the FFT to compute the spatial derivatives in the wave equation).
%       This allows infinite domain simulations to be computed using small
%       computational grids.
%
%       For a homogeneous medium the formulation is exact and the
%       time-steps are only limited by the effectiveness of the perfectly
%       matched layer. For a heterogeneous medium, the solution represents
%       a leap-frog pseudospectral method with a Laplacian correction that
%       improves the accuracy of computing the temporal derivatives. This
%       allows larger time-steps to be taken without instability compared
%       to conventional pseudospectral time-domain methods. The
%       computational grids are staggered both spatially and temporally.
%
%       An initial pressure distribution can be specified by assigning a
%       matrix (the same size as the computational grid) of arbitrary
%       numeric values to source.p0. A time varying pressure source can
%       similarly be specified by assigning a binary matrix (i.e., a matrix
%       of 1's and 0's with the same dimensions as the computational grid)
%       to source.p_mask where the 1's represent the pixels that form part
%       of the source. The time varying input signals are then assigned to
%       source.p. This must be the same length as kgrid.t_array and can be
%       a single time series (in which case it is applied to all source
%       elements), or a matrix of time series following the source elements
%       using MATLAB's standard column-wise linear matrix index ordering. A
%       time varying velocity source can be specified in an analogous
%       fashion, where the source location is specified by source.u_mask,
%       and the time varying input velocity is assigned to source.ux,
%       source.uy, and source.uz.
%
%       The pressure is returned as an array of time series at the sensor
%       locations defined by sensor.mask. This can be given either as a
%       binary grid (i.e., a matrix of 1's and 0's with the same dimensions
%       as the computational grid) representing the pixels within the
%       computational grid that will collect the data, or as a series of
%       arbitrary Cartesian coordinates within the grid at which the
%       pressure values are calculated at each time step via interpolation.
%       The Cartesian points must be given as a 3 by N matrix. The final
%       pressure field over the complete computational grid can also be
%       obtained using the output field_data. If no output is required, the
%       sensor input can be replaced with an empty array [].
%
%       If sensor.mask is given as a set of Cartesian coordinates, the
%       computed sensor_data is returned in the same order. If sensor.mask
%       is given as a binary grid, sensor_data is returned using MATLAB's
%       standard column-wise linear matrix index ordering. In both cases,
%       the recorded data is indexed as sensor_data(sensor_position, time).
%       For a binary sensor mask, the pressure values at a particular time
%       can be restored to the sensor positions within the computation grid
%       using unmaskSensorData.
%
%       By default, the recorded pressure field is passed directly to the
%       output arguments sensor_data and field_data. However, the particle
%       velocity can also be recorded by setting the optional input
%       parameter 'ReturnVelocity' to true. In this case, the output
%       arguments sensor_data and field_data are returned as structures
%       with the pressure and particle velocity appended as separate
%       fields. In one dimension, these fields are given by sensor_data.p,
%       sensor_data.ux, field_final.p, and field_final.ux.
%
%       kspaceFirstOrder1D may also be used for time reversal image
%       reconstruction by assigning the time varying pressure recorded over
%       an arbitrary sensor surface to the input field
%       sensor.time_reversal_boundary_data. This data is then enforced in
%       time reversed order as a time varying Dirichlet boundary condition
%       over the sensor surface given by sensor.mask. The boundary data
%       must be indexed as
%       sensor.time_reversal_boundary_data(sensor_position, time). If
%       sensor.mask is given as a set of Cartesian coordinates, the
%       boundary data must be given in the same order. An equivalent binary
%       sensor mask (computed using nearest neighbour interpolation) is
%       then used to place the pressure values into the computational grid
%       at each time step. If sensor.mask is given as a binary grid of
%       sensor points, the boundary data must be ordered using MATLAB's
%       standard column-wise linear matrix indexing. If no additional
%       inputs are required, the source input can be replaced with an empty
%       array [].
%
%       Acoustic attenuation compensation can also be included during time
%       reversal image reconstruction by assigning the absorption
%       parameters medium.alpha_coeff and medium.alpha_power and reversing
%       the sign of the absorption term by setting medium.alpha_sign = [-1,
%       1]. This forces the propagating waves to grow according to the
%       absorption parameters instead of decay. The reconstruction should
%       then be regularised by assigning a filter to medium.alpha_filter
%       (this can be created using getAlphaFilter).
%
%       Note: To run a simple reconstruction example using time reversal
%       (that commits the 'inverse crime' of using the same numerical
%       parameters and model for data simulation and image reconstruction),
%       the sensor_data returned from a k-Wave simulation can be passed
%       directly to sensor.time_reversal_boundary_data  with the input
%       fields source.p0 and source.p removed or set to zero.
%
% USAGE:
%       sensor_data = kspaceFirstOrder1D(kgrid, medium, source, sensor)
%       sensor_data = kspaceFirstOrder1D(kgrid, medium, source, sensor, ...) 
%
%       [sensor_data, field_data] = kspaceFirstOrder1D(kgrid, medium, source, sensor)
%       [sensor_data, field_data] = kspaceFirstOrder1D(kgrid, medium, source, sensor, ...) 
%
%       kspaceFirstOrder1D(kgrid, medium, source, [])
%       kspaceFirstOrder1D(kgrid, medium, source, [], ...) 
%
% INPUTS:
% The minimum fields that must be assigned to run an initial value problem
% (for example, a photoacoustic forward simulation) are marked with a *. 
%
%       kgrid*              - k-space grid structure returned by makeGrid
%                             containing Cartesian and k-space grid fields  
%       kgrid.t_array*      - evenly spaced array of time values [s] (set
%                             to 'auto' by makeGrid) 
%
%
%       medium.sound_speed* - sound speed distribution within the acoustic
%                             medium [m/s] 
%       medium.density*     - density distribution within the acoustic
%                             medium [kg/m^3] 
%       medium.alpha_power  - power law absorption exponent
%       medium.alpha_coeff  - power law absorption coefficient 
%                             [dB/(MHz^y cm)] 
%       medium.alpha_mode   - optional input to force either the absorption
%                             or dispersion terms in the equation of state
%                             to be excluded; valid inputs are
%                             'no_absorption' or 'no_dispersion' 
%       medium.alpha_filter - frequency domain filter applied to the
%                             absorption and dispersion terms in the
%                             equation of state 
%       medium.alpha_sign   - two element array used to control the sign of
%                             absorption and dispersion terms in the
%                             equation of state  
%
%
%       source.p0*          - initial pressure within the acoustic medium
%       source.p            - time varying pressure at each of the source
%                             positions given by source.p_mask 
%       source.p_mask       - binary grid specifying the positions of the
%                             time varying pressure source distribution
%       source.ux           - time varying particle velocity in the
%                             x-direction at each of the source positions
%                             given by source.u_mask 
%       source.u_mask       - binary grid specifying the positions of the
%                             time varying particle velocity distribution 
%
%
%       sensor.mask*        - binary grid or a set of Cartesian points
%                             where the pressure is recorded at each
%                             time-step  
%       sensor.time_reversal_boundary_data - time varying pressure
%                             enforced as a Dirichlet boundary condition
%                             over sensor.mask  
%       sensor.frequency response - two element array specifying the center
%                             frequency and percentage bandwidth of a
%                             frequency domain Gaussian filter applied to
%                             the sensor_data    
%
% Note: For heterogeneous medium parameters, medium.sound_speed and
% medium.density must be given in matrix form with the same dimensions as
% kgrid. For homogeneous medium parameters, these can be given as single
% numeric values. If the medium is homogeneous and velocity inputs or
% outputs are not required, it is not necessary to specify medium.density.
%
% OPTIONAL INPUTS:
%       Optional 'string', value pairs that may be used to modify the
%       default computational settings.
%
%       'CartInterp'- Interpolation mode used to extract the pressure when
%                     a Cartesian sensor mask is given. If set to 'nearest'
%                     and more than one Cartesian point maps to the same
%                     pixel, duplicated data points are discarded and
%                     sensor_data will be returned with less points than
%                     that specified by sensor.mask (default = 'nearest').
%       'CreateLog' - Boolean controlling whether the command line output
%                     is saved using the diary function with a date and
%                     time stamped filename (default = false). 
%       'DataCast'  - String input of the data type that variables are cast
%                     to before computation. For example, setting to
%                     'single' will speed up the computation time (due to
%                     the improved efficiency of fft and ifft for this
%                     data type) at the expense of a loss in precision.
%                     This variable is also useful for utilising GPU
%                     parallelisation through libraries such as GPUmat or
%                     AccelerEyesJacket by setting 'DataCast' to
%                     'GPUsingle' or 'gsingle' (default = 'off').
%       'DisplayMask' - Binary matrix overlayed onto the animated
%                     simulation display. Elements set to 1 within the
%                     display mask are set to black within the display
%                     (default = sensor.mask).
%       'LogScale'  - Boolean controlling whether the pressure field is log
%                     compressed before display (default = false). The data
%                     is compressed by scaling both the positive and
%                     negative values between 0 and 1 (truncating the data
%                     to the given plot scale), adding a scalar value
%                     (compression factor) and then using the corresponding
%                     portion of a log10 plot for the compression (the
%                     negative parts are remapped to be negative thus the
%                     default color scale will appear unchanged). The
%                     amount of compression can be controlled by adjusting
%                     the compression factor which can be given in place of
%                     the Boolean input. The closer the compression factor
%                     is to zero, the steeper the corresponding part of the
%                     log10 plot used, and the greater the compression (the
%                     default compression factor is 0.02).
%       'MovieArgs' - Settings for movie2avi. Parameters must be given as
%                     {param, value, ...} pairs within a cell array
%                     (default = {}).
%       'MovieName' - Name of the movie produced when 'RecordMovie' is set
%                     to true (default = 'date-time-kspaceFirstOrder2D').
%       'PlotFreq'  - The number of iterations which must pass before the
%                     simulation plot is updated (default = 10).
%       'PlotLayout'- Boolean controlling whether a four panel plot of the
%                     initial simulation layout is produced (initial
%                     pressure, sensor mask, sound speed, density)
%                     (default = false).
%       'PlotPML'   - Boolean controlling whether the perfectly matched
%                     layer is shown in the simulation plots. If set to
%                     false, the PML is not displayed (default = true).
%       'PlotScale' - [min, max] values used to control the scaling for
%                     imagesc (visualisation). If set to 'auto', a
%                     symmetric plot scale is chosen automatically for each
%                     plot frame.
%       'PlotSim'   - Boolean controlling whether the simulation iterations
%                     are progressively plotted (default = true).
%       'PMLAlpha'  - Absorption within the perfectly matched layer in
%                     Nepers per metre (default = 4).
%       'PMLInside' - Boolean controlling whether the perfectly matched
%                     layer is inside or outside the grid. If set to false,
%                     the input grids are enlarged by PMLSize before
%                     running the simulation (default = true). 
%       'PMLSize'   - size of the perfectly matched layer in voxels. To
%                     remove the PML, set the appropriate PMLAlpha to zero
%                     rather than forcing the PML to be of zero size
%                     (default = 20). 
%       'RecordMovie' - Boolean controlling whether the displayed image
%                     frames are captured and stored as a movie using
%                     movie2avi (default = false). 
%       'ReturnVelocity' - Boolean controlling whether the acoustic
%                     particle velocity at the positions defined by
%                     sensor.mask are also returned. If set to true, the
%                     output argument sensor_data is returned as a
%                     structure with the pressure and particle velocity
%                     appended as separate fields. In one dimension,
%                     these fields are given by sensor_data.p, and
%                     sensor_data.ux. The field data is similarly returned
%                     as field_data.p and field_data.ux.   
%       'Smooth'    - Boolean controlling whether source.p0,
%                     medium.sound_speed, and medium.density are smoothed
%                     using smooth before computation. 'Smooth' can either
%                     be given as a single Boolean value or as a 3 element
%                     array to control the smoothing of source.p0,
%                     medium.sound_speed, and medium.density,
%                     independently.  
%
% OUTPUTS:
% If 'ReturnVelocity' is set to false:
%       sensor_data - time varying pressure recorded at the sensor
%                     positions given by sensor.mask
%       field_data  - final pressure field
%
% If 'ReturnVelocity' is set to true:
%       sensor_data.p   - time varying pressure recorded at the sensor
%                         positions given by sensor.mask 
%       sensor_data.ux  - time varying particle velocity in the x-direction
%                         recorded at the sensor positions given by
%                         sensor.mask  
%       field_data.p    - final pressure field
%       field_data.ux   - final field of the particle velocity in the
%                         x-direction 
%
% ABOUT:
%       author      - Bradley Treeby and Ben Cox
%       date        - 22nd April 2009
%       last update - 10th February 2011
%       
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010, 2011 Bradley Treeby and Ben Cox
%
% See also fft, ifft, getframe, kspaceFirstOrder2D, kspaceFirstOrder3D,
% makeGrid, makeTime, movie2avi, smooth, unmaskSensorData 

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>.

% Known Issues
% - movie_frames storage variable is not preallocated
% - interp1 is not compatable with gsingle (use a binary sensor mask or set
%   'CartInterp' to 'nearest' as a work around)

% suppress mlint warnings that arise from using sub-scripts
%#ok<*NASGU>
%#ok<*COLND>
%#ok<*NODEF>
%#ok<*INUSL>

% start the timer and store the start time
start_time = clock;
tic

% =========================================================================
% DEFINE LITERALS
% =========================================================================

% minimum number of input variables
NUM_REQ_INPUT_VARIABLES = 4; 

% optional input defaults
CARTESIAN_INTERP_DEF = 'linear';
CREATE_LOG_DEF = false;
DATA_CAST_DEF = 'off';
DISPLAY_MASK_DEF = 'default';
LOG_SCALE_DEF = false;
LOG_SCALE_COMPRESSION_FACTOR_DEF = 0.02;
MOVIE_ARGS_DEF = {};
MOVIE_NAME_DEF = [getDateString '-kspaceFirstOrder1D'];
PLOT_FREQ_DEF = 10;
PLOT_LAYOUT_DEF = false;
PLOT_SCALE_DEF = [-1.1 1.1];
PLOT_SCALE_LOG_DEF = false;
PLOT_SIM_DEF = true;
PLOT_PML_DEF = true;
PML_ALPHA_DEF = 4;
PML_INSIDE_DEF = true;
PML_SIZE_DEF = 20;
RECORD_MOVIE_DEF = false;
RETURN_VELOCITY_DEF = false;
SMOOTH_P0_DEF = true;
SMOOTH_C0_DEF = false;
SMOOTH_RHO0_DEF = false;
USE_KSPACE_DEF = true;
USE_SG_DEF = true;

% set default movie compression
MOVIE_COMP_WIN = 'Cinepak';
MOVIE_COMP_LNX = 'None';
MOVIE_COMP_64B = 'None';

% set additional literals
MFILE = mfilename;
DT_WARNING_CFL = 0.5; 
ESTIMATE_SIM_TIME_STEPS = 10;
LOG_NAME = ['k-Wave-Log-' getDateString];
LOG_SCALE_COMPRESSION_FACTOR = 0.01;
PLOT_SCALE_WARNING = 5;

% =========================================================================
% CHECK INPUTS STRUCTURES AND OPTIONAL INPUTS
% =========================================================================

kspaceFirstOrder_inputChecking;

% forced exit for release B.0.1 inputs
if early_exit
    return
end

% =========================================================================
% UPDATE COMMAND LINE STATUS
% =========================================================================

disp(['  start time: ' datestr(start_time)]);
disp(['  dt: ' scaleSI(dt) 's, t_end: ' scaleSI(t_array(end)) 's, time steps: ' num2str(length(t_array))]);
disp(['  input grid size: ' num2str(kgrid.Nx) ' pixels (' scaleSI(kgrid.x_size) 'm)']);
disp(['  maximum supported frequency: ' scaleSI( kgrid.k_max * min(c(:)) / (2*pi) ) 'Hz']); 

% =========================================================================
% SMOOTH AND ENLARGE INPUT GRIDS
% =========================================================================

% smooth p0 distribution if required restoring the maximum magnitude
if smooth_p0 && ~time_rev
    disp('  smoothing p0 distribution...');  
    source.p0 = smooth(kgrid, source.p0, true);
end

% expand grid if the PML is set to be outside the input grid
if ~PML_inside
    
    % expand the computational grid
    disp('  expanding computational grid...');
    kgrid = makeGrid(kgrid.Nx + 2*PML_x_size, kgrid.dx);
           
    % expand the grid matrices
    expand_size = PML_x_size; %#ok<NASGU>
    kspaceFirstOrder_expandGridMatrices;
    clear expand_size;
        
    % update command line status
    disp(['  computation grid size: ' num2str(kgrid.Nx) ' pixels']);
end

if ~plot_PML
    % create indexes to allow the source input to be placed into the larger
    % simulation grid
    x1 = (PML_x_size + 1);
    x2 = kgrid.Nx - PML_x_size;
else
    % create indexes to place the source input exactly into the simulation
    % grid
    x1 = 1;
    x2 = kgrid.Nx;
end    

% select reference sound speed based on heterogeneity map
c_max = max(c(:));

% smooth c distribution if required
if smooth_c && numel(c) > 1
    disp('  smoothing sound speed distribution...');     
    c = smooth(kgrid, c);
end
    
% smooth rho0 distribution if required
if smooth_rho0 && numel(rho0) > 1
    disp('  smoothing density distribution...');     
    rho0 = smooth(kgrid, rho0);
end

% =========================================================================
% PREPARE STAGGERED COMPUTATIONAL GRIDS AND OPERATORS
% =========================================================================

% create the staggered grid
x_sg = kgrid.x + kgrid.dx/2;

% interpolate the values of the density at the staggered grid locations
if numel(rho0) > 1 && use_sg
    % rho0 is heterogeneous and staggered grids are used
    rho0_sg = interp1(kgrid.x, rho0, x_sg, '*linear');
    
    % set values outside of the interpolation range to original values 
    rho0_sg(isnan(rho0_sg)) = rho0(isnan(rho0_sg)); 
else
    % rho0 is homogeneous or staggered grids are not used
    rho0_sg = rho0;
end

% define the location of the perfectly matched layer within the grid
x0_min = kgrid.x(1) + PML_x_size*kgrid.dx;
x0_max = kgrid.x(end) - PML_x_size*kgrid.dx;

% set the PML attenuation over the pressure (regular) grid
ax = PML_x_alpha*(c_max/kgrid.dx)*((kgrid.x - x0_max)./(kgrid.x(end) - x0_max)).^4.*(kgrid.x >= x0_max)...
    + PML_x_alpha*(c_max/kgrid.dx)*((kgrid.x - x0_min)./(kgrid.x(1) - x0_min)).^4.*(kgrid.x <= x0_min);

% set the PML attenuation over the velocity (staggered) grid
ax_sg = PML_x_alpha*(c_max/kgrid.dx)*((x_sg - x0_max)./(kgrid.x(end) - x0_max)).^4.*(x_sg >= x0_max)...
    + PML_x_alpha*(c_max/kgrid.dx)*((x_sg - x0_min)./(kgrid.x(1) - x0_min)).^4.*(x_sg <= x0_min);

% precompute absorbing boundary condition operators
abc_x = exp(-ax_sg*dt/2);
abc_x_alt = exp(-ax*dt);

% define the modified first order k-space derivative operators (use
% optional input 'UsekSpace' to control usage)
if use_kspace
    % k-space
    ddx_k = 1i*kgrid.kx.*sinc(c_max*dt*kgrid.k/2);
else
    % pseudospectral
    ddx_k = 1i*kgrid.kx;
end

% define the staggered grid shift operators (use optional input 'UseSG' to
% control usage)
if use_sg
    shift_pos = exp(1i*kgrid.kx*kgrid.dx/2);
    shift_neg = exp(-1i*kgrid.kx*kgrid.dx/2);
else
    shift_pos = 1;
    shift_neg = 1;
end

% pre-shift variables used as frequency domain multipliers
ddx_k = ifftshift(ddx_k);
shift_pos = ifftshift(shift_pos);
shift_neg = ifftshift(shift_neg);

% clean up unused variables
clear ax* x0_min x0_max x_sg PML_x_alpha;

% =========================================================================
% PREPARE DATA MASKS AND STORAGE VARIABLES
% =========================================================================

kspaceFirstOrder_createAbsorptionAndStorageVariables;

% =========================================================================
% SET INITIAL CONDITIONS
% =========================================================================

% add the initial pressure to rho as a mass source
if isfield(source, 'p0')
    rhox = source.p0./c.^2;
else
    rhox = zeros(kgrid.Nx, 1);
end

% add p_source(t = t1) to rho as a mass source and create source variables
if p_source
           
    % scale the pressure source by 2*dt*c/dx divided by c^2 (because the
    % source is added to the density)
    if numel(c) == 1
        source.p = source.p .* (2*dt./(c*kgrid.dx));
    else
        % compute the scale parameter seperately for each source position
        % based on the sound speed at that position
        for p_index = 1:length(source.p(:,1))        
            source.p(p_index, :) = source.p(p_index, :) .* (2.*dt./(c(p_index).*kgrid.dx));
        end
    end
    
    % extract the initial source pressure and add to rho as a mass source
    rhox(ps_index) = rhox(ps_index) + source.p(p_element_index, 1);

end

% calculate pressure using an adiabatic equation of state
p = c.^2.*rhox;

% compute u(t = t1 - dt/2) based on the assumption u(dt/2) = -u(-dt/2)
% which forces u(t = t1) = 0
ux = dt./rho0_sg .* real(ifft(ddx_k .* shift_pos .* fft(p))) / 2;

% add u_source(t = t1) to u
if u_source
    
    % scale the velocity source by 2*dt*c/dx
    if numel(c) == 1
        source.ux = source.ux .* (2*c*dt./kgrid.dx);
    else
        % compute the scale parameter seperately for each source position
        % based on the sound speed at that position
        for u_index = 1:length(source.ux(:,1))
            source.ux(u_index, :) = source.ux(u_index, :) .* (2.*c(u_index).*dt./kgrid.dx);
        end
    end
    
    % extract the initial source velocity and add to u
    ux(us_index) = ux(us_index) + source.ux(u_element_index, 1);
    
end

% setup the time index variable
if ~time_rev
    index_start = 2;
    index_step = 1;
    index_end = length(t_array);  
else
    % reverse the order of the input data
    sensor.time_reversal_boundary_data = fliplr(sensor.time_reversal_boundary_data);    
    index_start = 1;
    index_step = 1;

    % stop one time point before the end so the last points are not
    % propagated
    index_end = length(t_array) - 1;      
end

% store p(t = t1) in forward mode
if use_sensor && ~time_rev
    
    switch extract_data_case
        case 1         
            % return velocity = false
            % binary sensor mask = false
            sensor_data(:, 1) = interp1(kgrid.x, p, sensor_x); 
            
        case 2
            % return velocity = false
            % binary sensor mask = true  
            sensor_data(:, 1) = p(sensor_mask_ind);
            
        case 3         
            % return velocity = true
            % binary sensor mask = false            
            sensor_data.p(:, 1) = interp1(kgrid.x, p, sensor_x); 
            sensor_data.ux(:, 1) = interp1(kgrid.x, ux, sensor_x);
            
        case 4         
            % return velocity = true
            % binary sensor mask = true
            sensor_data.p(:, 1) = p(sensor_mask_ind);
            sensor_data.ux(:, 1) = ux(sensor_mask_ind);
            
    end  
end

% =========================================================================
% PREPARE VISUALISATIONS
% =========================================================================

% pre-compute suitable axes scaling factor
if plot_layout || plot_sim
    [x_sc, scale, prefix] = scaleSI(max(kgrid.x));  %#ok<ASGLU>
end

% plot the simulation layout
if plot_layout
    figure;
    p0_plot = source.p0;
    subplot(2, 2, 1), plot(kgrid.x*scale, p0_plot);
    axis tight;
    title('Initial Pressure');
    if use_sensor
        subplot(2, 2, 2), plot(kgrid.x*scale, sensor.mask);
        axis tight;
        title('Sensor Mask');
    end
    subplot(2, 2, 3), plot(kgrid.x*scale, c);
    axis tight;
    title('Sound Speed');
    subplot(2, 2, 4), plot(kgrid.x*scale, rho0);
    axis tight;
    title('Density');
    xlabel(['(All horizontal axes in ' prefix 'm)']);
    clear p0_plot;
end

% initialise the figures used for animation
if plot_sim
    img = figure;
    if ~time_rev
        pbar = waitbar(0, 'Computing Pressure Field');
    else
        pbar = waitbar(0, 'Computing Time Reversed Field');
    end
end  

% set movie index variable
if record_movie
    
    % force getframe compatability with dual monitors
    movegui(img);    
    
    % define frame index
    frame_index = 1;
    
    % save the first frame if in forward mode
    if ~time_rev
        
        % create plot variable
        p_plot = p(x1:x2);
       
        % update plot
        plot(kgrid.x(x1:x2)*scale, p_plot);

        % update plot scale if set to automatic or log
        if plot_scale_auto || plot_scale_log
            kspaceFirstOrder_adjustPlotScale;
        end
        
        % add display mask onto plot
        if ~(strcmp(display_mask, 'default') || strcmp(display_mask, 'off'))
            hold on;
            plot(kgrid.x(x1:x2)*scale, display_mask(x1:x2).*(plot_scale(2) - plot_scale(1)) + plot_scale(1), 'k-');
            hold off
        end        
        
        % set plot options
        xlabel(['x-position [' prefix 'm]']);
        set(gca, 'YLim', plot_scale);        
        
        % force plot update
        drawnow;        
        
        % set background color to white
        set(gcf, 'Color', [1 1 1]);

        % save the movie frame
        movie_frames(frame_index) = getframe(gcf);

        % update frame index
        frame_index  = frame_index  + 1;
    end
end

% =========================================================================
% DATA CASTING
% =========================================================================

kspaceFirstOrder_dataCast;

% =========================================================================
% LOOP THROUGH TIME STEPS
% =========================================================================

% update command line status
disp(['  precomputation completed in ' scaleTime(toc)]);
disp('  starting time loop...');

% restart timing variable
tic;

% precompute fft of p
p_k = fft(p);

for t_index = index_start:index_step:index_end
    
    % enforce time reversal bounday condition
    if time_rev   
        
        % load pressure value and enforce as a Dirichlet boundary condition
        p(sensor_mask_ind) = sensor.time_reversal_boundary_data(:, t_index);

        % update p_k
        p_k = fft(p);

        % compute rhox using an adiabatic equation of state
        rhox_mod = p./(c.^2);
        rhox(sensor_mask_ind) = rhox_mod(sensor_mask_ind);
        
    end
    
    % calculate u(t+dt/2) from u(t-dt/2) and dp(t)/dx
    ux = abc_x .* (  abc_x.*ux - dt./rho0_sg .* real(ifft(ddx_k .* shift_pos .* p_k))  );

    % add in the velocity source term
    if u_source
        ux(us_index) = ux(us_index) + source.ux(u_element_index, t_index);
    end    

    % calculate du(t+dt/2)/dx from u(t+dt/2)
    dudx_k = ddx_k .* shift_neg .* fft(ux);  

    % calculate rho at t + dt
    rhox = abc_x_alt .* (  rhox - dt.*rho0 .* real(ifft(dudx_k))  );        

    % add in the pressure source term as a mass source
    if p_source
        rhox(ps_index) = rhox(ps_index) + source.p(p_element_index, t_index);
    end    

    switch equation_of_state
        case 'lossless'
            % compute p using an adiabatic equation of state
            p = c.^2.*rhox;
        case 'absorbing'
            % compute p using an absorbing equation of state 
            p = c.^2.*(  rhox + real(ifft( absorb_param.*dudx_k + dispers_param.*fft(rhox) ))  );
        case 'stokes'
            % compute p using stokes equation of state
            p = c.^2.*(  rhox + absorb_param .* real(ifft(dudx_k))  );
    end
    
    % precompute fft of p here so p can be modified for visualisation
    p_k = fft(p);    
    
    % extract required data
    if ~time_rev
        switch extract_data_case
            case 1         
                % return velocity = false
                % binary sensor mask = false
                sensor_data(:, t_index) = interp1(kgrid.x, p, sensor_x); 

            case 2
                % return velocity = false
                % binary sensor mask = true  
                sensor_data(:, t_index) = p(sensor_mask_ind);

            case 3         
                % return velocity = true
                % binary sensor mask = false            
                sensor_data.p(:, t_index) = interp1(kgrid.x, p, sensor_x); 
                sensor_data.ux(:, t_index) = interp1(kgrid.x, ux, sensor_x);


            case 4         
                % return velocity = true
                % binary sensor mask = true
                sensor_data.p(:, t_index) = p(sensor_mask_ind);
                sensor_data.ux(:, t_index) = ux(sensor_mask_ind);

        end          
    end     

    % estimate the time to run the simulation
    if t_index == ESTIMATE_SIM_TIME_STEPS
        disp(['  estimated simulation time ' scaleTime(toc*index_end/t_index) '...']);
    end      
    
    % plot data if required
    if plot_sim && rem(t_index, plot_freq) == 0  

        % update progress bar
        waitbar(t_index/length(t_array), pbar);
        drawnow;

        % ensure p is cast as a CPU variable and remove the PML from the
        % plot if required
        p_plot = double(p(x1:x2));   
       
        % update plot scale if set to automatic or log
        if plot_scale_auto || plot_scale_log
            kspaceFirstOrder_adjustPlotScale;
        end
        
        % update plot
        plot(kgrid.x(x1:x2)*scale, p_plot);

        % add display mask onto plot
        if ~(strcmp(display_mask, 'default') || strcmp(display_mask, 'off'))
            hold on;
            plot(kgrid.x(x1:x2)*scale, display_mask(x1:x2).*(plot_scale(2) - plot_scale(1)) + plot_scale(1), 'k-');
            hold off
        end        
        
        % set plot options
        xlabel(['x-position [' prefix 'm]']);
        set(gca, 'YLim', plot_scale);
        
        % force plot update
        drawnow;
        
        % save movie frames if required
        if record_movie

            % set background color to white
            set(gcf, 'Color', [1 1 1]);

            % save the movie frame
            movie_frames(frame_index) = getframe(gcf);

            % update frame index
            frame_index  = frame_index  + 1;

        end
    end    
end

% =========================================================================
% CLEAN UP
% =========================================================================

% store the final movie frame and then save the movie
if record_movie
    
    % update command line status
    disp('  saving movie file...');
    
    % create plot variable
    p_plot = double(p(x1:x2));       

    % update plot scale if set to automatic or log
    if plot_scale_auto || plot_scale_log
        kspaceFirstOrder_adjustPlotScale;
    end   
    
    % update plot
    plot(kgrid.x(x1:x2)*scale, p_plot);

    % add display mask onto plot
    if ~(strcmp(display_mask, 'default') || strcmp(display_mask, 'off'))
        hold on;
        plot(kgrid.x(x1:x2)*scale, display_mask(x1:x2).*(plot_scale(2) - plot_scale(1)) + plot_scale(1), 'k-');
        hold off
    end        

    % set plot options
    xlabel(['x-position [' prefix 'm]']);
    set(gca, 'YLim', plot_scale);

    % force plot update
    drawnow;
        
    % set background color to white
    set(gcf, 'Color', [1 1 1]);

    % save the movie frame
    movie_frames(frame_index) = getframe(gcf);
   
    % save movie file
    kspaceFirstOrder_saveMovieFile;
    
end

% clean up used figures
if plot_sim
    close(img);
    close(pbar);
end

% reset the indexing variables to allow original grid size to be maintained
if (~plot_PML && PML_inside)
    x1 = x1 - PML_x_size;
    x2 = x2 + PML_x_size;
elseif (plot_PML && ~PML_inside)
    x1 = x1 + PML_x_size;
    x2 = x2 - PML_x_size;   
end

% save the final pressure field if in time reversal mode
if time_rev
    sensor_data = p(x1:x2);
end

% save the final pressure and velocity fields and cast variables back to
% double 
if return_velocity
    field_data.p = double(p(x1:x2));
    field_data.ux = double(ux(x1:x2));
    if use_sensor
        sensor_data.p = double(sensor_data.p);
        sensor_data.ux = double(sensor_data.ux);         
    end
else
    field_data = p(x1:x2);
    if use_sensor
        sensor_data = double(sensor_data);
    end
end

% reorder the sensor points if a binary sensor mask was used for Cartesian
% sensor mask nearest neighbour interpolation
if use_sensor && reorder_data
    
    % update command line status
    disp('  reordering Cartesian measurement data...');
    
    if return_velocity
        
        % append the reordering data
        new_col_pos = length(sensor_data.p(1,:)) + 1;
        sensor_data.p(:, new_col_pos) = reorder_index;
        sensor_data.ux(:, new_col_pos) = reorder_index;

        % reorder based on the order_index
        sensor_data.p = sortrows(sensor_data.p, new_col_pos);
        sensor_data.ux = sortrows(sensor_data.ux, new_col_pos);

        % remove the reordering data
        sensor_data.p = sensor_data.p(:, 1:new_col_pos - 1);
        sensor_data.ux = sensor_data.ux(:, 1:new_col_pos - 1);
        
    else
        
        % append the reordering data
        new_col_pos = length(sensor_data(1,:)) + 1;
        sensor_data(:, new_col_pos) = reorder_index;

        % reorder based on the order_index
        sensor_data = sortrows(sensor_data, new_col_pos);

        % remove the reordering data
        sensor_data = sensor_data(:, 1:new_col_pos - 1); 
        
    end
end

% filter the recorded time domain pressure signals if transducer parameters
% are given
kspaceFirstOrder_filterSensorData;

% return empty sensor data if not used
if ~use_sensor
    sensor_data = [];
end

% update command line status
disp(['  computation completed in ' scaleTime(toc)]);

% switch off log
if create_log
    diary off;
end